<?php

namespace App\DataTables;

use App\Models\UsersKPI;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class UserKPISDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     * @return \Yajra\DataTables\EloquentDataTable
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $users=$query->select('users_kpis.id','users_kpis.kpi_id','departments.title AS department','users.name AS user_name','users_kpis.user_id','users_kpis.year','users_kpis.month','users_kpis.total_employee_score','users_kpis.created_at',DB::raw("DATE_FORMAT(CAST(CONCAT(users_kpis.year,'-',users_kpis.month,'-01') AS DATE),'%Y-%m')  AS `year_month`"))->has('KPI')->has('User')
            ->leftJoin('users','users.id','=','users_kpis.user_id')
            ->leftJoin('kpis','kpis.id','=','users_kpis.kpi_id')
            ->leftJoin('departments','departments.id','=','kpis.department_id')
            ;
        if(!PerUser('users_kpis.show_all')){
            $users=$users->whereHas('User',function($q){
                return$q->where('path','LIKE','%:'.auth()->id().'::%');
            });
        }
        if(request('search_by')=='created_at'&&request('start_date')&&request('end_date')){
            $users=$users->whereBetween(DB::raw("DATE_FORMAT(users_kpis.created_at,'%Y-%m-%d')"),[request('start_date'),request('end_date')]);
        }
        if(request('search_by')=='year_month'&&request('start_date')&&request('end_date')){
            $start_date=request('start_date');
            $end_date=request('end_date');
            $users=$users->whereRaw("DATE_FORMAT(CAST(CONCAT(users_kpis.year,'-',users_kpis.month,'-01') AS DATE),'%Y-%m')  BETWEEN DATE_FORMAT('$start_date','%Y-%m') AND DATE_FORMAT('$end_date','%Y-%m')");
        }
        return (new EloquentDataTable($users))
            ->editColumn('checkbox',function($kpi){
                return '<div class="custom-control custom-checkbox"><input type="checkbox"  class="custom-control-input user_kpi-checkbox" value="'.$kpi->id.'" id="selectCheckbox-'.$kpi->id.'"><label class="custom-control-label" for="selectCheckbox-'.$kpi->id.'"></label></div>';
            })
            ->filterColumn('user.name',function($query, $keyword){
                $query->whereHas('User',function($q)use($keyword){
                    return $q->where('name', 'LIKE', "%$keyword%");
                });
            })
            ->filterColumn('k_p_i.name',function($query, $keyword){
                $query->whereHas('KPI',function($q)use($keyword){
                    return $q->where('name', 'LIKE', "%$keyword%");
                });
            })
            ->filterColumn('year_month',function($query, $keyword){
                $query->whereRaw("CONCAT(users_kpis.year,'-',users_kpis.month) LIKE '%$keyword%'");
            })
            ->addColumn('action', function($model){
                $html='<div class="btn-group pull-right">';
                if(PerUser('users_kpis.edit')){
                    $html.='<a href="'.route('users_kpis.edit',['users_kpi'=>$model->id]).'" class="btn btn-sm btn-success"><span class="fadeIn animated bx bx-edit-alt"> </span></a>';
                }
                if(PerUser('users_kpis.destroy')){
                    $html.='<a href="#" class="btn btn-sm btn-danger delete-this" data-id="'.$model->id.'" data-url="'.route('users_kpis.destroy',['users_kpi'=>$model]).'"><span class="fadeIn animated bx bx-trash"> </span></a>';
                }
                $html.='</div>';
                return$html;
            })
                ->editColumn('created_at',function ($model){
                    return $model->created_at->format('Y-m-d H:i:s');
                })

            ->rawColumns(['status','checkbox','action'])
            ->setRowId('id');
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\Models\UsersKPI $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(UsersKPI $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('users_kpis')
            ->columns($this->getColumns())
            ->minifiedAjax('',null,['search_by'=>'$("#search_by").val()','start_date'=>'$("#start_date").val()','end_date'=>'$("#end_date").val()'])
//                    ->dom('Bfrtip')
            ->orderBy(1)
            ->pageLength(10)
            ->lengthMenu([10, 20, 50, 100, 150])
//                    ->buttons(
//                        Button::make('export'),
//                        Button::make('print'),
//                        Button::make('reset'),
//                        Button::make('reload')
//                    )
            ;
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns(): array
    {
        return [
            Column::make('checkbox')
                ->title('<div class="custom-control custom-checkbox"><input type="checkbox" class="custom-control-input" id="selectAllCheckbox"><label class="custom-control-label" for="selectAllCheckbox"></label></div>')
                ->searchable(false)
                ->exportable(false)
                ->printable(false)
                ->width('10px')
                ->orderable(false),
            Column::make('id'),
            Column::make('user_name','users.name')->title(__('Employee Name')),
            Column::make('department','departments.title'),
            Column::make('year_month'),
            Column::make('total_employee_score'),
            Column::make('created_at','users_kpis.created_at'),

            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename(): string
    {
        return 'UserUserKPIS_' . date('YmdHis');
    }
}
