<?php

namespace App\DataTables;

use App\Models\UsersKPI;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class UserKPISTotalReportDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     * @return \Yajra\DataTables\EloquentDataTable
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $users=$query->select('users_kpis.user_id','users.name AS user_name','parent.name AS parent_name','departments.title AS department_title','job_titles.title AS job_title',DB::raw("SUM(users_kpis.total_employee_score)/COUNT('X') AS total"))
        ->leftJoin('users','users.id','=','users_kpis.user_id')
        ->leftJoin('users AS parent','parent.id','=','users.parent_id')
        ->leftJoin('departments','departments.id','=','users.department_id')
        ->leftJoin('job_titles','job_titles.id','=','users.job_title_id')
        ;
        if(!PerUser('users_kpis.show_all')){
            $users=$users->whereHas('User',function($q){
                return$q->where('path','LIKE','%:'.auth()->id().'::%');
            });
        }
        if(request('start_date')&&request('end_date')){
            $start_date=request('start_date');
            $end_date=request('end_date');
            $users=$users->whereRaw("DATE_FORMAT(CAST(CONCAT(users_kpis.year,'-',users_kpis.month,'-01') AS DATE),'%Y-%m')  BETWEEN DATE_FORMAT('$start_date','%Y-%m') AND DATE_FORMAT('$end_date','%Y-%m')");
        }
        $users=$users->groupBy('user_id');
        return (new EloquentDataTable($users))
            ->editColumn('checkbox',function($kpi){
                return '<div class="custom-control custom-checkbox"><input type="checkbox"  class="custom-control-input kpi-checkbox" value="'.$kpi->id.'" id="selectCheckbox-'.$kpi->id.'"><label class="custom-control-label" for="selectCheckbox-'.$kpi->id.'"></label></div>';
            })
            ->filterColumn('total',function($query, $keyword){
                if(request('start_date')&&request('end_date')){
                    $start_date=request('start_date');
                    $end_date=request('end_date');
                    $query->where(DB::raw("(SELECT SUM(uk.total_employee_score)/COUNT('X') FROM users_kpis AS uk WHERE uk.user_id=users_kpis.user_id AND DATE_FORMAT(CAST(CONCAT(uk.year,'-',uk.month,'-01') AS DATE),'%Y-%m')  BETWEEN DATE_FORMAT('$start_date','%Y-%m') AND DATE_FORMAT('$end_date','%Y-%m') AND uk.deleted_at IS NULL)"), 'LIKE', "%$keyword%");
                }else{
                    $query->where(DB::raw("(SELECT SUM(uk.total_employee_score)/COUNT('X') FROM users_kpis AS uk WHERE uk.user_id=users.id)"), 'LIKE', "%$keyword%");
                }

            })


            ->rawColumns(['status','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\Models\UsersKPI $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(UsersKPI $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('users_kpis')
            ->columns($this->getColumns())
            ->minifiedAjax('',null,['start_date'=>'$("#start_date").val()','end_date'=>'$("#end_date").val()'])
//                    ->dom('Bfrtip')
            ->orderBy(1)
            ->pageLength(10)
            ->lengthMenu([10, 20, 50, 100, 150])
//                    ->buttons(
//                        Button::make('export'),
//                        Button::make('print'),
//                        Button::make('reset'),
//                        Button::make('reload')
//                    )
            ;
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns(): array
    {
        return [
//            Column::make('checkbox')
//                ->title('<div class="custom-control custom-checkbox"><input type="checkbox" class="custom-control-input" id="selectAllCheckbox"><label class="custom-control-label" for="selectAllCheckbox"></label></div>')
//                ->searchable(false)
//                ->exportable(false)
//                ->printable(false)
//                ->width('10px')
//                ->orderable(false),
            Column::make('user_name','users.name')->title(__('Employee Name')),
            Column::make('department_title','departments.title')->title(__('Department')),
            Column::make('job_title','job_titles.title')->title(__('Job Title')),
            Column::make('parent_name','parent.name')->title(__('Manage By')),
            Column::make('total','total')->title(__('Total (SUM(total)/count)')),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename(): string
    {
        return 'UserUserKPIS_' . date('YmdHis');
    }
}
