<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use App\User;
use App\Visitor;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Session;
use App\Rules\FilterEmailDomain;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {

        return Validator::make($data, [
                        'name' => 'required|string|max:255|unique:users',
                        'email' => ['required','string','email','max:255','unique:users'],
                        'password' => 'required|string|min:8|max:32|confirmed',
                        'agreement' => 'required|in:1',
                        'mobile' => 'nullable|min:6|max:20',
                    ],
                    [
                        'name.unique'     => trans('home.name-unique'),
                        'name.required'    => 'من فضلك ضع اسمك',
                        'email.required'   => 'من فضلك ضع ايميلك',
                        'email.unique'     => 'للاسف الايميل موجود من قبل',
                        'password.required'=> 'من فضلك ضع الباسورد',
                        'password.confirmed'=> 'كلمة المرور وتأكيد كلمة المرور غير متطابقين',
                        'password.min'     => 'الباسورد يجب ان يكون 8 حروف اوارقام',
                        'password.max'     => 'الباسورد يجب ان لا يزيد عن 32 حروف اوارقام',
                        'agreement.required'=>'يرجي الموافقة علي الشروط والاحكام',
                        'agreement.in'=>'يجب الموافقة علي الشروط والاحكام',
                        'mobile.min'=> ' يجب أن يكون عدد الارقام أكبر من  أو بساوي 6 ',
                        'mobile.max'=>' يجب أن يكون عدد الارقام   أقل  من  أو بساوي 20 ',
                    ]);

    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\User
     */
    protected function create(array $data)
    {

        $user = User::create([
                'type' => 'visitors',
                'name' => $data['name'],
                'email' => $data['email'],
                'password' => Hash::make($data['password']),
                'verifyToken' => Str::random(40),
                'agreement' => $data['agreement'],
            ]);
            Visitor::create(['user_id' => $user->id, 'mobile' => $data['mobile']]);
            $thisUser = User::findOrFail($user->id);
            $this->sendEmail($thisUser);
            return $user;
    }


    public function sendEmail($user)
    {
        $html=view('email.sendView',compact('user'))->render();
        sendGridEmailToUser($html,$user->email,$user->name,'Verified Email');
    }
}
