<?php

namespace App;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Sessions extends Model
{
    use SoftDeletes;
    protected $fillable = [
        'therapist_id',
        'visitor_id',
        'type',
        'start_time',
        'status',
        'rating',
        'color',
        'topic',
        'text_color',
        'session_date'
    ];
    protected $appends = ['session_link', 'user_rate', 'user_name'];


    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'session_date', 'start_time'
    ];

    public function therapist()
    {
        return $this->belongsTo(Therapist::class,'therapist_id')->with('user');
    }
    public function Payments()
    {
        return $this->belongsTo(Payments::class,'session_id','id');
    }


    public function visitor()
    {
        return $this->belongsTo(Visitor::class,'visitor_id','id')->with('user');
    }

    public function reviews()
    {
        return $this->hasMany(Reviews::class);
    }

    public function treatment_plans()
    {
        return $this->hasOne(TreatmentPlan::class,'session_id');
    }

    public function messages()
    {
        return $this->hasMany(Message::class,'session_id');
    }

    //therapist profile
    public static  function getSessionsJson ($therapistData)
    {
        $events = array();
        $date=Carbon::now()->toDateString();
        $sessionCalender=[];
        $allSessions = Sessions::where([ ['therapist_id',$therapistData->id] ])->whereRaw("DATE(start_time)>='$date'")->get();
        foreach ($allSessions as $sessionData)
        {
            $events['title'] = $sessionData->type == 30 ? "30 Min" : "60 Min";
            $events['event_id'] = $sessionData->id;
            $events['start'] = $sessionData->start_time;
            $events['type'] = $sessionData->type;
            //$events['color'] = '#fffff';//$sessionData->color;
            //$events['textColor'] = $sessionData->text_color;
            $events['textColor'] ='#fffff';

            if($sessionData->visitor_id){
                $events['color'] = '#ea8181';
            }else{
                $events['color'] = '#19ef0c';
            }
            $sessionCalender [] = $events;
        }
        return  $sessionCalender;
    }

    //therapist public
    public static  function getSessionsForClients ($id)
    {
        $events = array();
        $sessionCalender=[];
        $allSessions = Sessions::where([ ['therapist_id',$id],['start_time','>=',Carbon::now()->toDateTimeString()] ])->get();
        foreach ($allSessions as $sessionData)
        {
            $events['title'] = $sessionData->type == 30 ? "30 minute" : "60 minute";
            $events['event_id'] = $sessionData->id;
            $events['start'] = $sessionData->start_time;
            $events['type'] = $sessionData->type;
            $events['color'] = $sessionData->color;
            $events['textColor'] = $sessionData->text_color;
            $sessionCalender [] = $events;
        }
        return  $sessionCalender;
    }

    

    public static function getTherapistSessions($therapist_id, $end = null)
    {
        $from = Carbon::now()->toDateTimeString();
        if ($end == "week")
            $to = Carbon::now()->addWeek()->toDateTimeString();
        elseif ($end == "manth")
            $to = Carbon::now()->addMonth()->toDateTimeString();
        else
            $to = Sessions::max('start_time');

        return Sessions::where('therapist_id','=', $therapist_id)->whereBetween('start_time', [$from, $to])
            ->orderBy('start_time', 'ASC') ->get()
            ->groupBy(function($session) {
                return $session->start_time->format('Y-m-d');
            })->mapToGroups(function($sessions, $date) {
                return ['data' => ['date' => $date, 'sessions'=> $sessions]];
            });
    }

    public static function getTherapistSessionsHistory($therapist_id)
    {
        $from = Carbon::now()->toDateTimeString();
        return Sessions::where('therapist_id', $therapist_id)->where('start_time', '<',$from)->orderBy('start_time', 'ASC')->get()
            ->groupBy(function($session) {
                return $session->start_time->format('Y-m-d');
            })->mapToGroups(function($sessions, $date) {
                return ['data' => ['date' => $date, 'sessions'=> $sessions]];
            });
    }


    public static function getTherapistSessionsToday($therapist_id)
    {
        $from = Carbon::now()->toDateTimeString();
        return Sessions::where('therapist_id', $therapist_id)->where('start_time', '=',$from)->orderBy('start_time', 'ASC')->get()
            ->groupBy(function($session) {
                return $session->start_time->format('Y-m-d');
            })->mapToGroups(function($sessions, $date) {
                return ['data' => ['date' => $date, 'sessions'=> $sessions]];
            });
    }




    public static function getVisitorSessions($visitor_id, $end = null)
    {

        // $todayDate=Carbon::now()->format('Y-m-d');
        
        // $nextSessions=Sessions::where('visitor_id',$visitor->id)->whereNotIn('session_status',['cancelled','visitor_cancelled'])->whereRaw("DATE(start_time)>'$todayDate'")->paginate(15);
        // return $nextSessions;




        $from = Carbon::now()->toDateTimeString();
        if ($end == "week")
            $to = Carbon::now()->addWeek()->toDateTimeString();
        elseif ($end == "manth")
            $to = Carbon::now()->addMonth()->toDateTimeString();
        else
            $to = Sessions::max('start_time');

        return Sessions::where('visitor_id','=', $visitor_id)->whereBetween('start_time', [$from, $to])
            ->orderBy('start_time', 'ASC') ->paginate(15)
            ->groupBy(function($session) {
                return $session->start_time->format('Y-m-d');
            })->mapToGroups(function($sessions, $date) {
                return ['data' => ['date' => $date, 'sessions'=> $sessions]];
            });
    }

    public static function getVisitorSessionsHistory($visitor_id)
    {
        $from = Carbon::now()->toDateTimeString();
        return Sessions::where('visitor_id', $visitor_id)->where('start_time', '<',$from)->orderBy('start_time', 'ASC')->paginate(10)
            ->groupBy(function($session) {
                return $session->start_time->format('Y-m-d');
            })->mapToGroups(function($sessions, $date) {
                return ['data' => ['date' => $date, 'sessions'=> $sessions]];
            });
    }

    protected static function boot()
    {
        parent::boot();
        static::updated(function ($model) {
            log_admin_action('update',$model);
            if ($model->isDirty('start_time'))
            {
               SessionsTimesChanges::create([
                   'session_id'=>$model->id,
                   'session_from_date'=>$model->getOriginal('start_time'),
                   'session_to_date'=>$model->start_time,
               ]);
            }
        });
        static::created(function ($model) {
            log_admin_action('create',$model);
        });
        static::deleted(function ($model) {
            log_admin_action('delete',$model);
        });
    }

    //rand number
    // 8f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe754

    public function getSessionLinkAttribute(){
        return url('/sessionVideo/'.$this->id.'/8f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe7548f5aee3a-b565-4ab8-bfe3-4112311fe754');
    }

    public function getUserNameAttribute(){
        $visitor = \App\Visitor::find($this->visitor_id);
            if ($visitor) {
               $user = \App\User::find($visitor->user_id);
               if($user)
               {
                 return $user->name;
               }
            }
    }



    public function getUserRateAttribute(){
        
        if(\App\Reviews::where('session_id', $this->id)->where('visitor_id', $this->visitor_id)->exists())
        {
            return true;
        }
        else
        {
            return false;
        }

    }






}
