<?php

namespace App\DataTables;

use App\Models\KPI;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class KPISDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     * @return \Yajra\DataTables\EloquentDataTable
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query->select('kpis.id','kpis.name','kpis.status','kpis.created_at')))
            ->editColumn('checkbox',function($kpi){
                return '<div class="custom-control custom-checkbox"><input type="checkbox"  class="custom-control-input kpi-checkbox" value="'.$kpi->id.'" id="selectCheckbox-'.$kpi->id.'"><label class="custom-control-label" for="selectCheckbox-'.$kpi->id.'"></label></div>';
            })
            ->editColumn('status',function($kpi){
                return '<span class="'.config('app.statusBadgeClass')[$kpi->status].'">'.$kpi->status.'</span>';
            })
            ->addColumn('action', function($model){
                $html='<div class="btn-group pull-right">';
                if(PerUser('kpis.edit')){
                    $html.='<a href="'.route('kpis.edit',['kpi'=>$model->id]).'" class="btn btn-sm btn-success"><span class="fadeIn animated bx bx-edit-alt"> </span></a>';
                }
                if(PerUser('kpis.destroy')){
                    $html.='<a href="#" class="btn btn-sm btn-danger delete-this" data-id="'.$model->id.'" data-url="'.route('kpis.destroy',['kpi'=>$model]).'"><span class="fadeIn animated bx bx-trash"> </span></a>';
                }
                $html.='</div>';
                return$html;
            })
                ->editColumn('created_at',function ($model){
                    return $model->created_at->format('Y-m-d H:i:s');
                })

            ->rawColumns(['status','checkbox','action'])
            ->setRowId('id');
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\Models\KPI $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(KPI $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('kpis')
            ->columns($this->getColumns())
            ->minifiedAjax()
//                    ->dom('Bfrtip')
            ->orderBy(1)
            ->pageLength(10)
            ->lengthMenu([10, 20, 50, 100, 150])
//                    ->buttons(
//                        Button::make('export'),
//                        Button::make('print'),
//                        Button::make('reset'),
//                        Button::make('reload')
//                    )
            ;
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns(): array
    {
        return [
            Column::make('checkbox')
                ->title('<div class="custom-control custom-checkbox"><input type="checkbox" class="custom-control-input" id="selectAllCheckbox"><label class="custom-control-label" for="selectAllCheckbox"></label></div>')
                ->searchable(false)
                ->exportable(false)
                ->printable(false)
                ->width('10px')
                ->orderable(false),
            Column::make('id'),
            Column::make('name'),
            Column::make('status'),
            Column::make('created_at'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename(): string
    {
        return 'KPIS_' . date('YmdHis');
    }
}
